package easik.sketch.util.Export.Components;

import java.util.LinkedList;


/**
 * Class representing a path from the sketch.  Contains references to all edges associated in the 
 * path in a LinkedList.  Should only be used for exportation to a database.  
 * Does not provide functionality for graphical display.
 * 
 * @author Vera Ranieri 2006
 * @since 2006-05-22 Vera Ranieri
 * @version 2006-06-22 Vera Ranieri
 */
public class ExportPath {
	/**
	 * The edges involved in this path
	 */
	private LinkedList<ExportEdge> _edges;
	/**
	 * The name of this path
	 */
	private String _name;
	
	/**
	 * The domain of this path
	 */
	private String _domain;
	
	/**
	 * Constructor to initialize linkedlist of edges
	 * 
	 * @param n The name of the path
	 * @param d The domain of the path
	 * @since 2006-05-22, Vera Ranieri
	 */
	public ExportPath(String n, String d){
		_name = n;
		_edges = new LinkedList<ExportEdge>();
		_domain = d;
	}
	
	/**
	 * Getter method for the linked list of edges contained in this path.
	 * 
	 * @return The linked list of edges for this path
	 * @since 2006-05-22
	 */
	public LinkedList<ExportEdge> getEdges(){
		return _edges;
	}
	
	/**
	 * Adds an edge to the path.  Builds the path according to the order they are stored in 
	 * the XML file.
	 * 
	 * @param e The edge to be added to the end of the path
	 * @since 2006-05-22
	 */
	public void addEdge(ExportEdge e){
		_edges.add(e);
	}
	
	/**
	 * Returns the unique name of this path, created by the edges contained in this path.
	 * <br>
	 * <b>Example:</b> If a path contains edges e1, e2, e3 in that order, the name of the path is 
	 * taken to be e1;e2;e3 .
	 *  
	 * @since 2006-05-22
	 * @return The name of this path
	 */
	public String getName(){
		return _name;
	}
	
	/**
	 * Returns the domain of the first edge in the edge collection of the path, 
	 * and thus the domain of the path.
	 * @since 2006-05-22
	 * @return The domain of the path
	 */
	public String getDomain(){
		return _domain;
	}
	
	/**
	 * Returns the codomain of the last edge in the edge collection of the path,
	 *  and thus the codomain of the path.
	 * @since 2006-05-23
	 * @return The codomain of the path
	 */
	public String getCoDomain(){
		return ((ExportEdge)_edges.getLast()).getTarget();
	}
	
	
	/**
	 * Determines the tables along a path, and formats them into a string.
	 * 
	 * @param path The path for which the tables must be determined
	 * @return A string for SQL containing a list of all tables involved in the path.
	 * @since 2006-06-16, Vera Ranieri
	 */
	public static String getAllTablesString(ExportPath path){
		LinkedList edges = path.getEdges();
		String allTables = new String();
		
		int size = edges.size();
		
		for(int i = 0; i< size; i++){
			allTables += ((ExportEdge)edges.get(i)).getSource() + ", ";
		}
		
		allTables += ((ExportEdge)edges.getLast()).getTarget() +" ";
		return allTables;
		
	}
	/**
	 * Determines the initial condition, as required by a <b>BEFORE DELETE</b> procedure.  Once initial condition is determined,
	 * all other conditions are then determined.
	 * 
	 * @param path The path for which the delete conditions need to be determined
	 * @return String of the delete conditions
	 * @since 2006-06-20, Vera Ranieri
	 */
	public static String getDelConditions(ExportPath path){
		
		String conditions, source;
		source = path.getDomain();
		conditions =  source+ "." + source + "_id=id ";
		
		conditions += getConditions(path);
		return conditions;
	}
	
	/**
	 * Gets the conditions for a path.  These conditions are determined by matching unique ids along the path.
	 * 
	 * @param path The path for which an element needs to be deleted from the codomain.
	 * @return A string of the conditions.
	 * @since 2006-06-22, Vera Ranieri
	 */
	public static String getConditions(ExportPath path){
		
		String conditions, source, target;
		LinkedList edges = (LinkedList)path.getEdges();
		
		conditions = "";
		
		int size = edges.size();
		
		for(int i = 0; i < size ; i++){
			source = ((ExportEdge)edges.get(i)).getSource();
			target = ((ExportEdge)edges.get(i)).getTarget(); 
			conditions += "AND " + source+ "." +target +"_id=" +
							target+"."+target+"_id ";
		}
		
		return conditions;
	}
}
